<?php

namespace Application\Services\Mail;

use Application\Exception\MailerException;
use Application\Model\User;
use Laminas\Mail\Transport\Smtp;
use Laminas\Mime\Message;
use Laminas\Mime\Mime;
use Laminas\Mime\Part;
use Laminas\ServiceManager\ServiceLocatorInterface;
use Laminas\View\Model\ViewModel;
use Laminas\View\Renderer\PhpRenderer;
use Laminas\View\Resolver\TemplateMapResolver;

/**
 * Class GenericMailerInterface
 * @package Application\Services\Mail
 */
abstract class GenericMailerInterface
{
    /**
     * @var $serviceLocator ServiceLocatorInterface
     */
    protected $serviceLocator;

    /**
     * @var $view PhpRenderer
     */
    protected $view;

    /**
     * @var $templateMapResolver TemplateMapResolver
     */
    protected $templateMapResolver;

    /**
     * @var $mailTransport Smtp
     */
    protected $mailTransport;

    /**
     * @var $htmlPart Part
     */
    private $htmlPart;

    /**
     * @var $mimeMessage Message
     */
    private $mimeMessage;


    /**
     * @var $mailMessage \Laminas\Mail\Message
     */
    private $mailMessage;

    /**
     * GenericMailerInterface constructor.
     * @param ServiceLocatorInterface $serviceLocator
     */
    public function __construct(ServiceLocatorInterface $serviceLocator)
    {
        $this->serviceLocator = $serviceLocator;

        $this->view = new PhpRenderer();
        $this->templateMapResolver = $this->serviceLocator->get('MailTemplateMapResolver');
        $this->view->setResolver($this->templateMapResolver);
        $this->mailTransport = $this->serviceLocator->get('MailTransport');
    }

    /**
     * @param ViewModel $viewModel
     * @return Part
     */
    protected function getHtmlPart(ViewModel $viewModel, $new = false)
    {
        if (!isset($this->htmlPart) || $new)
        {
            $rendered = $this->view->render($viewModel);
            $this->htmlPart = new Part($rendered);
            $this->htmlPart->setEncoding(Mime::ENCODING_QUOTEDPRINTABLE);
            $this->htmlPart->type = 'text/html';
        }
        return $this->htmlPart;
    }

    /**
     * @param bool $new
     * @return Message
     */
    protected function getMimeMessage($new = false)
    {
        if (!isset($this->mimeMessage) || $new)
            $this->mimeMessage = new Message();
        return $this->mimeMessage;
    }

    /**
     * @param bool $new
     * @return \Laminas\Mail\Message
     */
    protected function getMailMessage($new = false)
    {
        if (!isset($this->mailMessage))
        {
            $this->mailMessage = new \Laminas\Mail\Message();
            $this->mailMessage->setEncoding("UTF-8");
        }
        
        return $this->mailMessage;
    }

    /**
     * @throws MailerException
     */
    protected function dispatchMail()
    {
        if ($this->getMailMessage()->getFrom()->count() <= 0)
            throw new MailerException("Email from address not defined");

        if ($this->getMailMessage()->getTo()->count() <= 0)
            throw new MailerException("Email to address not defined");

        if (!strlen($this->getMailMessage()->getSubject()))
            throw new MailerException("Email subject not defined");

        if (!sizeof($this->getMimeMessage()->getParts()))
            throw new MailerException("Email mime parts not defined");

        $this->mailTransport->send($this->mailMessage);
    }

    /**
     * @return mixed
     */
    protected function getDefaultFrom()
    {
        $config = $this->serviceLocator->get('config');
        return $config['smtp_service']['default_from'];
    }

    /**
     * @return mixed
     */
    protected function getHelpdeskEmail()
    {
        $config = $this->serviceLocator->get('config');
        return $config['smtp_service']['helpdesk_email'];
    }

    abstract public function sendMail(User $user, $data);
}