<?php

namespace Application\Services\Mail\Cron;

use Application\Model\EmailTemplate;
use Application\Model\GuideAssignment;
use Application\Model\GuideAssignmentTable;
use Application\Model\GuideTimeSheet;
use Application\Model\GuideTimeSheetsTable;
use Application\Model\TaskAssignment;
use Application\Model\TaskAssignmentTable;
use Application\Model\TaskType;
use Application\Model\TaskTypesTable;
use Application\Model\TourType;
use Application\Model\TourTypesTable;
use Application\Model\User;
use Application\Model\UserTable;
use Application\Services\Mail\GenericMailerInterface;
use Laminas\Db\ResultSet\ResultSet;
use Laminas\View\Model\ViewModel;

class TimeSheetWeeklyNotesMailer extends GenericMailerInterface
{
    /**
     * @var User[]
     */
    private $usersList = null;

    /**
     * @var TaskType[]
     */
    private $taskTypeList = null;

    /**
     * @var TourType[]
     */
    private $tourTypeList = null;

    /**
     * @return User[]|null
     */
    private function getUsers()
    {
        if ($this->usersList === null)
        {
            $this->usersList = [];

            /**
             * @var $usersTable UserTable
             */
            $usersTable = $this->serviceLocator->get('Application\Model\UserTable');

            /**
             * @var $resultSet ResultSet
             */
            $resultSet = $usersTable->getLiveUsers();

            /**
             * @var $user User
             */
            while ($user = $resultSet->current())
            {
                $this->usersList[$user->id] = $user;
                $resultSet->next();
            }

        }

        return $this->usersList;
    }

    /**
     * @param $id
     * @return User
     */
    private function getUser($id)
    {
        $usersList = $this->getUsers();

        if (array_key_exists($id, $usersList) && $usersList[$id] instanceof User)
        {
            return $usersList[$id];
        }

        $noUser = new User();
        $noUser->name = sprintf('Unknown User (id: %d)', $id);
        return $noUser;
    }

    /**
     * @return TaskType[]
     */
    private function getTaskTypeList()
    {
        if ($this->taskTypeList == null)
        {
            $this->taskTypeList = [];

            /**
             * @var $taskTypeTable TaskTypesTable
             */
            $taskTypeTable = $this->serviceLocator->get('Application\Model\TaskTypeTable');

            /**
             * @var $resultSet ResultSet
             */
            $resultSet = $taskTypeTable->fetchAll();

            /**
             * @var $taskType TaskType
             */
            while ($taskType = $resultSet->current())
            {
                $this->taskTypeList[$taskType->id] = $taskType;
                $resultSet->next();
            }
        }

        return $this->taskTypeList;
    }

    /**
     * @return TourType[]|array
     */
    private function getTourTypeList()
    {
        if ($this->tourTypeList == null)
        {
            $this->tourTypeList = [];

            /**
             * @var $tourTypeTable TourTypesTable
             */
            $tourTypeTable = $this->serviceLocator->get('Application\Model\TourTypeTable');

            /**
             * @var $resultSet ResultSet
             */
            $resultSet = $tourTypeTable->fetchAll();

            /**
             * @var $tourType TourType
             */
            while ($tourType = $resultSet->current())
            {
                $this->tourTypeList[$tourType->id] = $tourType;
                $resultSet->next();
            }
        }

        return $this->tourTypeList;
    }

    /**
     * @param $id
     * @return TourType
     */
    private function getTourType($id)
    {
        $tourTypes = $this->getTourTypeList();

        if (array_key_exists($id, $tourTypes) && $tourTypes[$id] instanceof TourType)
        {
            return $tourTypes[$id];
        }

        $tourType = new TourType();
        $tourType->name = 'Unknown Tour Type';
        return $tourType;
    }

    public function executeCronAction()
    {
        /**
         * @var $taskAssignmentTable TaskAssignmentTable
         */
        $taskAssignmentTable = $this->serviceLocator->get('Application\Model\TaskAssignmentTable');

        /**
         * @var $guideTimeSheetsTable GuideTimeSheetsTable
         */
        $guideTimeSheetsTable = $this->serviceLocator->get('Application\Model\GuideTimeSheetsTable');


        /**
         * @var $guideAssignmentsTable GuideAssignmentTable
         */
        $guideAssignmentsTable = $this->serviceLocator->get('Application\Model\GuideAssignmentTable');


        $startDate = strtotime("-7 days");
        $endDate   = strtotime("-1 days");

        /**
         * FIXME: hard coded
         *
         * Indexed by organization id
         */
        $timesheetnotes = [ 1 => [], 2 => []];

        /**
         * @var $guideTimeSheetsResultSet ResultSet
         */
        $guideTimeSheetsResultSet = $guideTimeSheetsTable->fetchCurrent($startDate, $endDate, true);

        /**
         * @var $guideTimeSheet GuideTimeSheet
         */
        while ($guideTimeSheet = $guideTimeSheetsResultSet->current())
        {
            if ($guideTimeSheet->notes == '')
            {
                $guideTimeSheetsResultSet->next();
                continue;
            }

            $note = new \stdClass();
            $note->guide = $this->getUser($guideTimeSheet->guide_id);
            $note->date = strtotime($guideTimeSheet->assignmentdate);
            $note->notes = $guideTimeSheet->notes;

            $timesheetnotes[$guideTimeSheet->organization_id][] = $note;

            $guideTimeSheetsResultSet->next();
        }



        /**
         * @var $taskAssignmentsResultSet ResultSet
         */
        $taskAssignmentsResultSet = $taskAssignmentTable->fetchCurrent($startDate, $endDate, null, 1);

        /**
         * FIXME: hard coded
         *
         * Indexed by organization id
         */
        $tasksheetnotes = [
            'private'        => [ 1 => [], 2 => []],
            'public'         => [ 1 => [], 2 => []],
            'administrative' => [ 1 => [], 2 => []]
        ];


        /**
         * @var $taskAssignment TaskAssignment
         */
        while ($taskAssignment = $taskAssignmentsResultSet->current())
        {
            if ($taskAssignment->tasktype != 1 && $taskAssignment->tasktype != 5)
            {
                $taskAssignmentsResultSet->next();
                continue;
            }

            try
            {
                /**
                 * @var $tourType TourType
                 */
                $tourType = $this->getTourType($taskAssignment->tourtype);

                /**
                 * @var $guideAssignmentsResultSet ResultSet
                 */
                $guideAssignmentsResultSet = $guideAssignmentsTable->fetchAllByAssignmentIds([$taskAssignment->id], 1);

                /**
                 * @var $guideAssignment GuideAssignment
                 */
                while ($guideAssignment = $guideAssignmentsResultSet->current())
                {

                    if ($guideAssignment->notes == '')
                    {
                        $guideAssignmentsResultSet->next();
                        continue;
                    }

                    $taskSheetNote = new \stdClass();
                    $taskSheetNote->guide = $this->getUser($guideAssignment->guide_id);
                    $taskSheetNote->date = strtotime($taskAssignment->assignmentdate);
                    $taskSheetNote->notes = $guideAssignment->notes;
                    $taskSheetNote->tourname = $taskAssignment->tourname;
                    $taskSheetNote->tourtype = $tourType;
                    $taskSheetNote->time = $taskAssignment->getTaskTime();

                    if ($taskAssignment->tasktype == 5)
                    {
                        $tasksheetnotes['administrative'][$taskAssignment->organization_id][] = $taskSheetNote;
                    } else if ($taskAssignment->tasktype == 1 && $taskAssignment->tourtype == 1)
                    {
                        $tasksheetnotes['public'][$taskAssignment->organization_id][] = $taskSheetNote;
                    } else if ($taskAssignment->tasktype == 1)
                    {
                        $tasksheetnotes['private'][$taskAssignment->organization_id][] = $taskSheetNote;
                    }

                    $guideAssignmentsResultSet->next();
                }

            } catch (\Exception $exception)
            {
            }

            $taskAssignmentsResultSet->next();
        }

        foreach ([1, 2] as $id)
        {
            $this->sendMail(null, [
                'organization' => $id == 1 ? 'The Constitutional' : 'Spirits of 76',
                'start_date' => $startDate,
                'end_date' => $endDate,
                'timesheet_notes' => $timesheetnotes[$id],
                'task_notes' => [
                    'private'         => $tasksheetnotes['private'][$id],
                    'public'          => $tasksheetnotes['public'][$id],
                    'administrative'  => $tasksheetnotes['administrative'][$id]
                ]
            ]);
        }

        return [null, null];
    }

    /**
     * @param User|null $user
     * @param $mailData
     */
    public function sendMail(User $user = null, $mailData)
    {

        $config = $this->serviceLocator->get('config');
        $config = $config['smtp_service'];

        /**
         * @var $messageFooter EmailTemplate
         */
        $messageFooter = $this->serviceLocator->get('EmailFooterFactory');
        $messageFooter = $messageFooter->content;

        /**
         * @var $messageHeader EmailTemplate
         */
        $messageHeader = $this->serviceLocator->get('EmailHeaderFactory');
        $messageHeader = $messageHeader->content;


        $viewModel = new ViewModel();
        $viewModel->setTemplate('messaging/admin-weekly-notes')
            ->setVariables([
                'organization'    => $mailData['organization'],
                'start_date'      => $mailData['start_date'],
                'end_date'        => $mailData['end_date'],
                'timesheet_notes' => $mailData['timesheet_notes'],
                'task_notes'      => $mailData['task_notes'],
                'messageHeader'   => $messageHeader,
                'messageFooter'   => $messageFooter
            ]);


        $this->getMimeMessage(true)->addPart($this->getHtmlPart($viewModel, true));

        $mail = $this->getMailMessage(true);
        $mail->setFrom($config['default_from']);

        if (APPLICATION_ENVIRONMENT == 'production')
        {
            $mail->setTo($config['default_from']);
        } else
        {
            $mail->setTo($config['debug_from']);
        }

        $mail->setSubject(sprintf("Time Sheet Notes for %s-%s",
            date('l, F j, Y', $mailData['start_date']),
            date('l, F j, Y', $mailData['end_date'])
        ));
        $mail->setBody($this->getMimeMessage());

        $this->dispatchMail();

    }
}
