<?php

namespace Application\Services\Auth;

use Application\Model\Admin;
use Laminas\Authentication\AuthenticationService;
use Laminas\EventManager\AbstractListenerAggregate;
use Laminas\EventManager\EventManagerInterface;
use Laminas\Mvc\MvcEvent;
use Laminas\ServiceManager\ServiceManager;


class Listener extends AbstractListenerAggregate
{
    /**
     * @var $authService AuthenticationService
     */
    private $authService;

    /**
     * @var $serviceLocator ServiceManager
     */
    private $serviceLocator;


    public function __construct(ServiceManager $serviceLocator, AuthenticationService $service)
    {
        $this->serviceLocator = $serviceLocator;
        $this->authService = $service;
    }

    /**
     * We attach a checkAcl function to the Auth Services Event Listener so that every request that comes through
     * must go through a standard set of checks.
     *
     * @param EventManagerInterface $eventManager
     */
    public function attach(EventManagerInterface $eventManager)
    {
        $this->listeners[] = $eventManager->attach(MvcEvent::EVENT_ROUTE, [$this, 'checkAcl'], -1000);
    }

    /**
     * Check basic things like if a user is logged in and whether they have basic rights to the
     * route and actions they're trying to perform
     *
     * see the config file in config\autoload\global.php
     *
     * @param MvcEvent $event
     * @throws \Exception
     */
    public function checkAcl(MvcEvent $event)
    {
        $config = $this->serviceLocator->get('config');

        if (array_key_exists('disable_system', $config) && $config['disable_system'])
        {
            $event->getRouteMatch()
                ->setParam('controller', 'Application\Controller\Index')
                ->setParam('action', 'systemMaintenance');
            return;
        }

        /**
         * There must be a list of api restrictions for the rest-api route
         */
        if (!array_key_exists('api-restrictions', $config) || !is_array($config['api-restrictions']))
        {
            throw new \Exception("Configuration invalid");
        }

        $api_restrictions = $config['api-restrictions'];

        /**
         * all of the methods GET, PUT and POST MUST be defined
         */
        if (!array_key_exists('GET', $api_restrictions)  || !is_array($api_restrictions['GET']) ||
            !array_key_exists('PUT', $api_restrictions)  || !is_array($api_restrictions['PUT']) ||
            !array_key_exists('POST', $api_restrictions) || !is_array($api_restrictions['POST'])
        ){
            throw new \Exception('Configuration invalid');
        }

        $matchedRouteName = $event->getRouteMatch()->getMatchedRouteName();

        $routeMatch = $event->getRouteMatch();
        $route_controller = $routeMatch->getParam('controller');
        $route_action = $routeMatch->getParam('action');
        $request = $event->getRequest();


        /**
         * Prevent access to certain areas if you aren't logged in
         */
        if (!$this->authService->hasIdentity())
        {
            switch ($matchedRouteName)
            {
                case 'rest-api' :

                    $controllersList = $api_restrictions[$request->getMethod()];
                    $bypassList = array_key_exists($route_controller, $controllersList) ? $controllersList[$route_controller] : [];

                    if ($this->canBypass($bypassList, $request)) return;

                    $event->getRouteMatch()
                        ->setParam('controller', 'API\Controller\InvalidIdentity')
                        ->setParam('action', 'go');
                    break;
                case 'application' :
                    $event->getRouteMatch()
                        ->setParam('controller', 'Application\Controller\Index')
                        ->setParam('action', 'index');
                    break;
                case 'wstr/default' :
                    if ($route_controller === 'Application\Controller\Partials')
                    {
                        $event->getRouteMatch()
                            ->setParam('controller', 'Application\Controller\Login')
                            ->setParam('action', 'index');
                    }
                    break;

            }

            return;
        }

        /**
         * If we're accessing anything other than the keep-alive URL, save a last active timestamp
         */

        if ($route_controller !== 'Application\Controller\Login' && $route_action !== 'ping')
        {
            $_SESSION['_ctime'] = time();
        }

        /**
         * Here we can set up a per-route/request and action denial for users that are logged in
         * It's not a full real ACL, but it works well enough
         */

        try
        {
            /**
             * @var $admin Admin
             */
            $admin = $this->serviceLocator->get('ActiveAdmin');
        } catch (\Exception $e)
        {
            $admin = new Admin();
            $admin->enabled = false;
            $admin->user_id = null;
        }

        if ($matchedRouteName === 'rest-api')
        {
            /**
             * Delete Requests
             *
             * All delete requests must be accompanied by an active admin object
             */
            if ($request->getMethod() === 'DELETE')
            {
                if (!$admin->enabled)
                {
                    $event->getRouteMatch()
                        ->setParam('controller', 'API\Controller\InvalidIdentity')
                        ->setParam('action', 'admin-required');
                    return;
                }
            }

            /**
             * For any method other than GET, see if the listed controller is registered
             *
             * If there are any query params or post params defined in in the config that will
             * allow bypassing admin restrictions (meaning the controller will handle security for
             * those specific requests), if the provided query/post params tests pass, allow the
             * controller to handle the request.  Otherwise, restrict to admins only
             *
             */
            if (array_key_exists($route_controller, $api_restrictions[$request->getMethod()]) && !$admin->enabled)
            {
                $bypassList = $api_restrictions[$request->getMethod()][$route_controller];

                if ($this->canBypass($bypassList, $request)) return;

                $event->getRouteMatch()
                    ->setParam('controller', 'API\Controller\InvalidIdentity')
                    ->setParam('action', 'admin-required');
                return;
            }
        }
    }

    /**
     * Test to see if we can bypass base security for a particular controller and action
     * The controllers and actions MUST implement their own security tests if they're configured
     * to bypass basic identity and admin requests
     *
     * @param $bypassList
     * @param $request
     * @return bool
     * @throws \Exception
     */
    public function canBypass($bypassList, $request)
    {
        $can_bypass = false;

        if (is_array($bypassList) && sizeof($bypassList))
        {
            foreach ($bypassList as $bypassType => $checks)
            {
                foreach ($checks as $check_key => $test)
                {
                    switch ($bypassType)
                    {
                        case 'postExceptions' : $requestParam = $request->getPost($check_key, null); break;
                        case 'getExceptions'  : $requestParam = $request->getQuery($check_key, null); break;
                        default :
                            throw new \Exception("Configuration Invalid");
                    }

                    list ($must_have_identity, $test_type, $test_value) = preg_split('/:/', $test);

                    if ($must_have_identity === 'true' && !$this->authService->hasIdentity())
                        return false;

                    switch ($test_type)
                    {
                        case 'equals' :
                                if ($requestParam === $test_value) $can_bypass = true;
                                else return false;
                            break;
                        case 'function' :
                                if (@$test_type($requestParam)) $can_bypass = true;
                                else return false;
                            break;
                    }
                }
            }
        }
        return $can_bypass;
    }
}