<?php

namespace API\Controller;

use API\Exception\DeletedEntryNotificationException;
use API\Exception\SystemErrorException;
use Application\Exception\EntryNotFoundException;
use Application\Model\Availability;
use Application\Model\AvailabilityTable;
use Application\Model\GuideAvailability;
use Application\Model\GuideAvailabilityTable;
use Application\Services\DatePickerFormatter;
use Laminas\Db\ResultSet\ResultSet;
use Laminas\Mvc\Controller\AbstractRestfulController;
use Laminas\View\Model\JsonModel;

class AvailabilitiesController extends AbstractRestfulController
{
    public function get($id)
    {
        try
        {

            /**
             * @var $availabilityTable AvailabilityTable
             */
            $availabilityTable = $this->serviceLocator->get('Application\Model\AvailabilityTable');

            /**
             * @var $availability Availability
             */
            $availability = $availabilityTable->fetchOne($id);
            $availability->startdate = DatePickerFormatter::toDatePickerDate($availability->startdate);
            $availability->enddate   = DatePickerFormatter::toDatePickerDate($availability->enddate);

            return new JsonModel($availability->toArray());

        } catch (EntryNotFoundException $notFoundException)
        {
            return new \API\Exception\EntryNotFoundException($this->response,
                "UNABLE TO FIND ENTRY");
        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }
    }

    public function getList()
    {
        try
        {

            $guide_id = $this->request->getQuery('guide_id', null);

            /**
             * @var $availabilityTable AvailabilityTable
             */
            $availabilityTable = $this->serviceLocator->get('Application\Model\AvailabilityTable');

            /**
             * @var $resultSet ResultSet
             */
            $resultSet = $availabilityTable->getList($guide_id);

            $return = [];

            /**
             * @var $availability Availability
             */
            while ($availability = $resultSet->current())
            {
                $availability->startdate = DatePickerFormatter::toDatePickerDate($availability->startdate);
                $availability->enddate   = DatePickerFormatter::toDatePickerDate($availability->enddate);

                $return[] = $availability->toArray();

                $resultSet->next();
            }

            return new JsonModel($return);

        } catch (EntryNotFoundException $notFoundException)
        {
            return new \API\Exception\EntryNotFoundException($this->response,
                "UNABLE TO FIND ENTRY");
        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }
    }

    public function create($data)
    {
        try
        {
            /**
             * @var $availabilityTable AvailabilityTable
             */
            $availabilityTable = $this->serviceLocator->get('Application\Model\AvailabilityTable');

            $availability = new Availability();
            $availability->exchangeArray($data);

            $availability->createdate = date("Y-m-d H:i:s");
            $availability->modifydate = date("Y-m-d H:i:s");

            $availability->startdate = DatePickerFormatter::toSqlDate($availability->startdate);
            $availability->enddate   = DatePickerFormatter::toSqlDate($availability->enddate);

            $availabilityTable->saveOne($availability);

            $availability->startdate = DatePickerFormatter::toDatePickerDate($availability->startdate);
            $availability->enddate   = DatePickerFormatter::toDatePickerDate($availability->enddate);

            /**
             * @var $guideAvailabilityTable GuideAvailabilityTable
             */
            $guideAvailabilityTable = $this->serviceLocator->get('Application\Model\GuideAvailabilityTable');

            $oneDay = new \DateInterval("P1D");
            $timezone = new \DateTimeZone("America/New_York");
            $startTime = new \DateTime($availability->startdate, $timezone);
            $endTime = new \DateTime($availability->enddate, $timezone);

            while ($startTime <= $endTime)
            {
                if ((int) $startTime->format("w") != (int) $availability->dayofweek)
                {
                    $startTime->add($oneDay);
                    continue;
                }

                $guideAvailability = new GuideAvailability();
                $guideAvailability->createdate = $availability->createdate;
                $guideAvailability->modifydate = $availability->modifydate;
                $guideAvailability->guide_id = $availability->guide_id;
                $guideAvailability->starttime = $availability->starttime;
                $guideAvailability->endtime = $availability->endtime;
                $guideAvailability->custom = 0;
                $guideAvailability->availabilitydate = $startTime->format("Y-m-d");
                $guideAvailabilityTable->saveOne($guideAvailability);

                $startTime->add($oneDay);
            }

            return new JsonModel($availability->toArray());

        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }
    }

    public function update($id, $data)
    {
        try
        {

            /**
             * @var $availabilityTable AvailabilityTable
             */
            $availabilityTable = $this->serviceLocator->get('Application\Model\AvailabilityTable');

            /**
             * @var $old Availability
             */
            $old = $availabilityTable->fetchOne($id);

            $availability = new Availability();

            $availability->exchangeArray($data);

            $availability->createdate = $old->createdate;
            $availability->modifydate = date('Y-m-d H:i:s');

            $availability->startdate = DatePickerFormatter::toSqlDate($availability->startdate);
            $availability->enddate = DatePickerFormatter::toSqlDate($availability->enddate);

            $availabilityTable->saveOne($availability);

            $availability->startdate = DatePickerFormatter::toDatePickerDate($availability->startdate);
            $availability->enddate = DatePickerFormatter::toDatePickerDate($availability->enddate);

            /**
             * @var $guideAvailabilityTable GuideAvailabilityTable
             */
            $guideAvailabilityTable = $this->serviceLocator->get('Application\Model\GuideAvailabilityTable');

            $guideAvailabilityTable->deleteLinked($availability->createdate);

            $oneDay = new \DateInterval("P1D");
            $timezone = new \DateTimeZone("America/New_York");
            $startTime = new \DateTime($availability->startdate, $timezone);
            $endTime = new \DateTime($availability->enddate, $timezone);


            while ($startTime <= $endTime)
            {

                if ((int) $startTime->format("w") != (int) $availability->dayofweek)
                {
                    $startTime->add($oneDay);
                    continue;
                }

                $guideAvailability = new GuideAvailability();
                $guideAvailability->createdate = $availability->createdate;
                $guideAvailability->modifydate = $availability->modifydate;
                $guideAvailability->guide_id = $availability->guide_id;
                $guideAvailability->starttime = $availability->starttime;
                $guideAvailability->endtime = $availability->endtime;
                $guideAvailability->custom = 0;
                $guideAvailability->availabilitydate = $startTime->format("Y-m-d");
                $guideAvailabilityTable->saveOne($guideAvailability);

                $startTime->add($oneDay);
            }

            return new JsonModel($availability->toArray());

        } catch (EntryNotFoundException $notFoundException)
        {
            return new \API\Exception\EntryNotFoundException($this->response,
                "UNABLE TO FIND ENTRY");
        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }
    }

    public function delete($id)
    {
        try {

            /**
             * @var $availabilityTable AvailabilityTable
             */
            $availabilityTable = $this->serviceLocator->get('Application\Model\AvailabilityTable');

            $availability = $availabilityTable->fetchOne($id);

            $availabilityTable->deleteOne($id);

            /**
             * @var $guideAvailabilityTable GuideAvailabilityTable
             */
            $guideAvailabilityTable = $this->serviceLocator->get('Application\Model\GuideAvailabilityTable');

            $guideAvailabilityTable->deleteLinked($availability->createdate);

            return new JsonModel(new DeletedEntryNotificationException($this->response,
                "Availability Deleted"));


        } catch (EntryNotFoundException $notFoundException)
        {
            return new \API\Exception\EntryNotFoundException($this->response,
                "UNABLE TO FIND ENTRY");
        } catch (\Exception $exception)
        {
            return new JsonModel(new SystemErrorException($this->response, $exception));
        }

    }
}
